define([
    'underscore',
    'jquery',
    'backbone',
    'moment',
    'modules/new-appointment-request/views/direct/clinics/layout-view',
    'modules/new-appointment-request/views/direct/layout-view',
    'modules/new-appointment-request/views/direct/preferred-date/selected-date-time',
    'modules/new-appointment-request/resources/direct/clinics/collection',
    'modules/new-appointment-request/tests/helpers/clinics-helper',
], function(_, $, Backbone, moment, View, DetailsView, DateView, Collection, helpers) {
    'use strict';


    /**
     * As a veteran directly booking an appointment in VAR
     * I want to see a list which clinics are available for a given type of care at my selected location
     * So that I can schedule an appointment
     */
    describe('Specification for VAR-7748', function() {
        var region;

        beforeEach(function() {
            region = helpers.setupTestingRegion();
        });

        afterEach(helpers.resetTestingRegion);

        /**
         * When a user views the appointment details section of the New appointment/Request form,
         * after selecting “Schedule Myself” the following fields are shown:
         *
         * 1.   Clinic, required field
         *
         * 1.1. Displays all clinics matching the stop code definition for the selected
         *      type of care and the selected location, where the clinic profile flag for allow
         *      direct scheduling = yes, and the profile flag for display in VAR = yes.
         */
        describe('Clinic Display', function() {


            /**
             * 1.2. If only one clinic is offered, the form displays the single clinic (and its location)
             * as static text.
             * 1.2.1. Display format:
             * 1.2.1.1. Clinic Name
             * 1.2.1.2. Clinic Friendly Location Name
             */
            describe('Single Clinic', function() {
                var view;
                var clinics;

                beforeEach(function() {
                    var options;
                    helpers.collectionFetchSpy();

                    options = {
                        customMessages: new Backbone.Model(),
                        model: helpers.layoutModelFactory(),
                    };
                    region.show(new View(options));
                    view = region.currentView;

                    clinics = [helpers.getMockClinicList()[0]];

                    view.collection = new Collection(clinics, {parse: true});
                });

                it('generates the correct text', function() {
                    var $clinic;
                    var $label;

                    view.showClinics();
                    $clinic = view.$('#clinic');
                    $label = $clinic.find('.label-text > p');

                    expect($clinic.text()).toContain(clinics[0].clinicName);
                    expect($label.text()).toBe(clinics[0].clinicFriendlyLocationName);
                });
            });


            /**
             * 1.3. If more than one clinic option is available, each option is listed as a selection in a radio button list.
             * 1.3.1. Radio button display format:
             * 1.3.1.1. Clinic Name
             * 1.3.1.2. Clinic Friendly Location Name
             *
             * 1.3.2. The list is ordered alphabetically by clinic name
             */
            describe('Multiple Clinics', function() {
                var view;
                var clinics;

                beforeEach(function() {
                    var options;
                    helpers.collectionFetchSpy();

                    options = {
                        customMessages: new Backbone.Model(),
                        model: helpers.layoutModelFactory(),
                    };
                    region.show(new View(options));
                    view = region.currentView;

                    clinics = helpers.getMockClinicList();

                    view.collection = new Collection(clinics, {parse: true});
                });

                it('generates the correct text', function() {
                    var id;
                    var $input;
                    var $label;
                    var name;
                    var locationName;

                    view.showClinics();
                    id = '#clinic-' + clinics[0].clinicId;
                    $input = view.$(id);
                    $label = $input.parent();

                    name = $label.find('span > span > span').text();
                    locationName = $label.find('p').text();

                    expect(name).toContain(clinics[0].clinicName);
                    expect(locationName).toBe(clinics[0].clinicFriendlyLocationName);
                });

                it('orders the clinics alphabetically by name', function() {
                    var labels;
                    var labelText;

                    view.showClinics();
                    labels = view.$('label > .label-text');
                    labelText = _.map(labels, function(el) {
                        return helpers.cleanString(el.innerText);
                    });

                    console.log(labelText);
                    expect(labelText[0].localeCompare(labelText[1])).toBe(-1);
                    expect(labelText[1].localeCompare(labelText[2])).toBe(-1);
                });
            });
        });


        /**
         * 1.4. When a user selects a clinic, the app checks for available slots in the next 90 days.
         */
        describe('Appointment Slots', function() {


            describe('With Slots', function() {
                var view;
                var clinics;
                var slots;

                beforeEach(function() {
                    var options;
                    var layout;

                    helpers.collectionFetchSpy();

                    options = {
                        customMessages: new Backbone.Model(),
                        model: helpers.layoutModelFactory(),
                    };
                    region.show(new View(options));
                    layout = region.currentView;

                    clinics = helpers.getMockClinicList();

                    layout.collection = new Collection(clinics, {parse: true});
                    layout.showClinics();
                    view = layout.getRegion('containerRegion').currentView;

                    spyOn(view, 'updateFDNS   el').and.callThrough();
                    spyOn(view, 'showErrorMessage').and.callFake(_.noop);
                });

                /**
                 * 1.4.1. Loading spinner displays if needed while the app determines available slots.
                 */
                // TEST REMOVED.  This is handled by the framework.

                /**
                 * Conclusion for 1.4.1
                 */
                it('hides the spinner after fetch', function() {
                    view.$('input:first').click();
                    view.fetchSlotsSuccess(new Backbone.Collection());
                    expect($('.loaderModalMask').length).toBe(0);
                });

                /**
                 * 1.4.1.1. If slots are found, the app moves focus to the next field.
                 */
                xit('changes focus when slots are found', function() {
                    // TODO This test can not pass when written like this
                    // TODO Because the region that shows the next section does not exist
                    // TODO Create a test for preferred date that checks that it give the label focus

                    var fakeData = slots[0];
                    fakeData.appointmentTime = helpers.getAppointmentSlots();

                    view.$('input:first').click();
                    view.fetchSlotsSuccess(new Backbone.Collection([fakeData]));

                    expect(view.updateFDNS   el).toHaveBeenCalled();
                    expect($(':focus').tagName).toBe('label');
                });
            });


            describe('Without Slots', function() {
                var view;

                beforeAll(function() {
                    $('.ui-popup-screen').each(function() {
                        $(this).remove();
                    });
                });

                beforeEach(function() {
                    var options;
                    var layout;
                    var clinics;

                    helpers.collectionFetchSpy();

                    options = {
                        customMessages: new Backbone.Model(),
                        model: helpers.layoutModelFactory(),
                    };
                    region.show(new View(options));
                    layout = region.currentView;
                    clinics = helpers.getMockClinicList();

                    layout.collection = new Collection(clinics, {parse: true});

                    layout.showClinics();
                    view = layout.getRegion('containerRegion').currentView;
                    spyOn(view, 'updateFDNS   el').and.callThrough();
                });

                /**
                 * 1.4.1.1. If slots are found, the app moves focus to the next field.
                 */
                it('creates an error modal', function() {
                    var modal;

                    view.checkClinic(view.model);
                    view.slots.trigger('fetch:success', new Backbone.Collection());

                    modal = $('.ui-popup-screen');

                    expect(modal.length).toBe(1);
                });

                /**
                 * 1.4.1.2.1. Modal Header: No Available Appointments
                 * 1.4.1.2.2. Modal Text: Unfortunately, there are no available time slots at this clinic.
                 *            Please select another clinic or change your scheduling method to
                 *            "Submit a Request to VA Scheduler.”
                 * 1.4.1.2.3. OK button
                 */
                it('displays the correct text in the error model', function() {
                    var title;
                    var text;

                    view.checkClinic(view.model);
                    view.slots.trigger('fetch:success', new Backbone.Collection());

                    title = $('.ui-title')[0].innerText;
                    text = $('.ui-content > p')[0].innerText;

                    expect(title).toBe('No Available Appointments');
                    expect(text).toBe('Unfortunately, there are no available time slots at this clinic. Please select another clinic or change your scheduling method to "Submit a Request to VA Scheduler.”');
                });

                /**
                 * 1.4.1.2.3.1. When OK is chosen, the user's focus is returned to the clinic field label.
                 */
                xit('restores focus correctly after okay is clicked', function() {
                    // TODO I have to do this from an outer view.  The area it is trying to restore focus too does not exist in this limited view.
                    view.checkClinic(view.model);
                    view.slots.trigger('fetch:success', new Backbone.Collection());
                    $('button').click();

                    expect($(':focus').length).toBe(1);
                });
            });
        });


        /**
         * 2. Reason for Appointment
         * 2.1. Required
         * 2.2. Multi line text box, 150 characters, display character countdown in right corner
         */
        describe('Reason For Appointment', function() {
            var view;

            beforeEach(function() {
                var model;
                var options;
                helpers.collectionFetchSpy();

                model = helpers.layoutModelFactory();
                model.set('typeOfCare', new Backbone.Model({
                    id: '323',
                }));
                model.set('facility', new Backbone.Model({
                    institutionCode: '523',
                }));
                options = {
                    model: model,
                };

                region.show(new DetailsView(options));
                view = region.currentView;
                view._showPurposeSectionView();
            });

            it('is a required field', function() {
                var required = view.$('.validation-required');
                expect(required.length).toBe(1);
            });

            it('limits the text area to 150 chars', function() {
                var text = view.$('textarea');
                var maxLength = text.attr('maxlength');

                expect(maxLength).toBe('150');
            });

            // Not part of the spec but still important to test
            xit('sanitizes the text before storing it in the model', function() {
                expect(false).toBe(null);
            });
        });


        /**
         * 3. Preferred Date (required, Calendar date picker control; date range must be between today + 1 and today + 90)
         */
        describe('Preferred Date', function() {
            var view;

            beforeEach(function() {
                var model;
                var options;

                helpers.collectionFetchSpy();
                spyOn(View.prototype, 'getTimeSlots').and.callFake(function() {
                    return new Backbone.Model({appointmentTimeSlot: new Backbone.Collection()});
                });

                spyOn(DateView.prototype, 'addFormValidationRules').and.callFake(_.noop);

                model = helpers.layoutModelFactory();
                model.set('typeOfCare', new Backbone.Model({
                    id: '323',
                }));
                model.set('facility', new Backbone.Model({
                    institutionCode: '523',
                }));
                options = {
                    model: model,
                };

                region.show(new DetailsView(options));
                view = region.currentView;
                view._showDateTimeLayoutView();
            });

            it('is a required field', function() {
                var required = view.$('.validation-required');
                expect(required.length).toBe(1);
            });

            it('sets the range between 1 and 90 days', function() {
                var range = view.$('.form-input-datepicker-description');
                var regDate = /\d{2}\/\d{2}\/\d{4}/g;
                var text = range.text();
                var matches = text.match(regDate);
                var start;
                var end;
                var expectedStart;
                var expectedEnd;

                expect(matches.length).toBe(2);

                start = matches[0];
                end = matches[1];

                expectedStart = moment().add(1, 'day')
                    .format('MM/DD/YYYY');
                expectedEnd = moment().add(90, 'day')
                    .format('MM/DD/YYYY');

                expect(start).toBe(expectedStart);
                expect(end).toBe(expectedEnd);
                // TODO i need to figure out how to check the picker is enforcing the range.
            });
        });


        /**
         * 4. Show Availability button
         */
        describe('Show Availability button', function() {

            var view;

            beforeEach(function() {
                var modal;
                var options;

                helpers.collectionFetchSpy();
                spyOn(View.prototype, 'getTimeSlots').and.callFake(function() {
                    return new Backbone.Model({appointmentTimeSlot: new Backbone.Collection()});
                });

                spyOn(DateView.prototype, 'addFormValidationRules').and.callFake(_.noop);
                modal = helpers.layoutModelFactory();
                modal.set('typeOfCare', new Backbone.Model({id: '323'}));
                modal.set('facility', new Backbone.Model({institutionCode: '523'}));
                options = {model: modal};

                region.show(new DetailsView(options));
                view = region.currentView;
                view._showDateTimeLayoutView();
            });

            /**
             * 4.1.   Disabled when Preferred Date is null;
             * 4.1.1. Disabled button is read as disabled by screen reader and additional information is also read:
             *        "Enter a preferred date to enable Show Availability button"
             */
            describe('Before Preferred Date Selected', function() {
                it('is disabled', function() {
                    var button = view.$('#show-availability-btn');
                    var classList = button[0].classList;
                    var isDisabled = classList.contains('ui-state-disabled');

                    expect(isDisabled).toBeTruthy();
                });

                it('has the correct screen reader text', function() {
                    var button = view.$('#show-availability-btn');
                    var ariaLabel = button.attr('aria-label');
                    expect(ariaLabel).toBe('Enter a preferred date to enable Show Availability button');
                });
            });


            /**
             * 4.1.2. When Preferred Date is valued button becomes enabled; selecting displays Select
             *        Date/Time accordion, instructional text based on availability, and selected appointment
             *        date/time static text, as follows:
             */
            xdescribe('After Preferred Date is Selected', function() {


                /**
                 * 4.1.3.   If Preferred Date is available for scheduling, display:
                 * 4.1.3.1. Select Date/Time label (required); note month labels should read Month and Year (Month YYYY)
                 * 4.1.3.2. Instructional Text: The following times are available on your preferred date:
                 * 4.1.3.3. Accordion control displaying preferred date as expanded with time options viewable but none selected
                 * 4.1.3.4. Show All Dates (button) is displayed
                 *
                 * 4.1.3.4.1.   If the user selects the Show All Dates option when their preferred date is available, then:
                 * 4.1.3.4.1.1. Focus moves to the Select Date/Time label.
                 * 4.1.3.4.1.2. Instructional Text is updated to read "Showing all available dates."
                 * 4.1.3.4.1.3. The preferred date remains expanded and all other available dates
                 *              in the next 90 days are shown
                 *
                 * 4.1.3.4.1.4. Separate accordions are shown for each month, months are labeled;
                 *              times are not visible for dates other than the preferred date
                 *
                 * 4.1.3.4.1.5. Show All Dates button is hidden.
                 */
                describe('Preferred Date Available', function() {
                    it('shows the data time label in the required format', function() {
                        expect(false).toBe(null);
                    });

                    it('displays the correct instruction text', function() {
                        expect(false).toBe(null);
                    });

                    it('has an accordion with the time options listed', function() {
                        expect(false).toBe(null);
                    });

                    it('it does not preselect a time', function() {
                        expect(false).toBe(null);
                    });

                    it('has the show all dates button displayed', function() {
                        expect(false).toBe(null);
                    });

                    it('shows the correct instructional text if show all dates is clicked', function() {
                        expect(false).toBe(null);
                    });

                    it('preferred dates does not collapse when show all is clicked', function() {
                        expect(false).toBe(null);
                    });

                    it('it shows all other dates once the show all is clicked', function() {
                        expect(false).toBe(null);
                    });

                    it('shows times if a different according are clicked', function() {
                        expect(false).toBe(null);
                    });
                });


                /**
                 * 4.1.4.   If Preferred Date is not available for scheduling, then display:
                 * 4.1.4.1. Select Date/Time label (required)
                 * 4.1.4.2. Instructional Text: No appointments are available on your preferred date.
                 *          These are the closest available dates. Select "Show All Dates" to see all date options.
                 * 4.1.4.3. Accordion control displaying the three dates (collapsed) that are closest to
                 *          the preferred date (before or after); if two or more dates "tie" for the closest date,
                 *          then show the date(s) that are earlier/before the preferred date over the date that is after
                 * 4.1.4.4. Show All Dates (button) is enabled
                 *
                 * 4.1.4.4.1.   If the user selects the Show All Dates option when their preferred date is not available, then:
                 * 4.1.4.4.1.1. Focus moves to the Select Date/Time label
                 * 4.1.4.4.1.2. Instructional text below the label is updated to read "No appointments are available on
                 *              your preferred date. Showing all available dates."
                 * 4.1.4.4.1.3. All available dates are shown (collapsed accordions)
                 *
                 * 4.1.4.4.1.4. Separate accordions are shown for each Month; Months are labeled.
                 * 4.1.4.4.1.5. Show All Dates button is hidden.
                 * 4.2.         Dynamic text based on Date/Time selected is displayed below the accordion date selectors:
                 *              You are scheduling an appointment for <date> @ <time>.
                 */
                describe('Preferred Date Not Available', function() {
                    it('has the required labels', function() {
                        expect(false).toBe(null);
                    });

                    it('displays the correct instructional text', function() {
                        expect(false).toBe(null);
                    });

                    it('shows the three closest dates to preferred dates', function() {
                        expect(false).toBe(null);
                    });

                    it('keeps the dates accordion collapsed', function() {
                        expect(false).toBe(null);
                    });

                    it('breaks ties with selecting the earliest dates', function() {
                        expect(false).toBe(null);
                    });

                    it('has show all dates button enabled', function() {
                        expect(false).toBe(null);
                    });

                    it('shows the correct instructional texts after show all is selected', function() {
                        expect(false).toBe(null);
                    });

                    it('all accordions are collapsed', function() {
                        expect(false).toBe(null);
                    });

                    it('after clicking show all the show button is hidden', function() {
                        expect(false).toBe(null);
                    });

                    it('updates the display texts when a time is selected', function() {
                        expect(false).toBe(null);
                    });
                });


                /**
                 * 4.3. If Preferred Date is changed after "Show Availability" has been selected,
                 *      the Select Date/Time and related text options are nullified and hidden until
                 *      Show Availability is selected again for the new Preferred Date.
                 */
                describe('Preferred Date changed after Time Select', function() {
                    it('resets show availability until the it is selected again', function() {
                        expect(false).toBe(null);
                    });
                });


                /*
                 * 4.4. Screen readers read accordion control actions ("Expand", "Collapse") and indicate selected date
                 *      and time values.
                 */
                describe('Screen Reader', function() {
                    it('reads the expand values', function() {
                        expect(false).toBe(null);
                    });

                    it('reads the collapse values', function() {
                        expect(false).toBe(null);
                    });
                });
            });
        });
    });
});
